# gophian -- tools to help with Debianizing Go software
# Copyright (C) 2025 Maytham Alsudany <maytham@debian.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.

import re

import click

ANSI_COLOURS = [
    "black",
    "red",
    "green",
    "yellow",
    "blue",
    "magenta",
    "cyan",
    "white",
    "bright_black",
    "bright_red",
    "bright_green",
    "bright_yellow",
    "bright_blue",
    "bright_magenta",
    "bright_cyan",
    "bright_white",
    "reset",
]
HEX_COLOUR_REGEX = re.compile(r"^#([a-fA-F0-9]{2})([a-fA-F0-9]{2})([a-fA-F0-9]{2})$")


def parse_colour(name: str, colour_string: str) -> str | int | tuple[int, int, int]:
    if colour_string in ANSI_COLOURS:
        return colour_string
    try:
        if match := HEX_COLOUR_REGEX.match(colour_string):
            [r_str, g_str, b_str] = match.groups()
            r = int(r_str, 16)
            g = int(g_str, 16)
            b = int(b_str, 16)
            return (r, g, b)
        colour_int = int(colour_string)
        if colour_int >= 0 and colour_int <= 255:
            return colour_int
    except Exception:
        pass
    click.echo(
        click.style("Error: ", bold=True, fg="red")
        + f"{name}: Invalid colour value: '{colour_string}'"
    )
    raise click.Abort()
