// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ash/wm/overview/birch/birch_chip_context_menu_model.h"

#include "ash/birch/coral_util.h"
#include "ash/constants/ash_pref_names.h"
#include "ash/resources/vector_icons/vector_icons.h"
#include "ash/session/session_controller_impl.h"
#include "ash/shell.h"
#include "ash/shell_delegate.h"
#include "ash/strings/grit/ash_strings.h"
#include "ash/wm/overview/birch/birch_bar_controller.h"
#include "ash/wm/overview/overview_utils.h"
#include "components/prefs/pref_service.h"
#include "ui/base/l10n/l10n_util.h"
#include "ui/base/models/menu_separator_types.h"
#include "ui/views/controls/menu/menu_types.h"

namespace ash {
namespace {

// Returns the pref service to use for Birch bar prefs.
PrefService* GetPrefService() {
  return Shell::Get()->session_controller()->GetPrimaryUserPrefService();
}

}  // namespace

BirchChipContextMenuModel::BirchChipContextMenuModel(
    ui::SimpleMenuModel::Delegate* delegate,
    BirchSuggestionType chip_type)
    : ui::SimpleMenuModel(delegate),
      sub_menu_model_(std::make_unique<BirchBarContextMenuModel>(
          delegate,
          BirchBarContextMenuModel::Type::kExpandedBarMenu)) {
  auto add_hide_suggestion_item = [&]() {
    AddItemWithIcon(
        base::to_underlying(CommandId::kHideSuggestion),
        l10n_util::GetStringUTF16(IDS_ASH_BIRCH_HIDE_THIS_SUGGESTION),
        CreateIconForMenuItem(kSystemTrayDoNotDisturbIcon));
  };

  switch (chip_type) {
    case BirchSuggestionType::kWeather:
      AddItemWithIcon(
          base::to_underlying(CommandId::kHideWeatherSuggestions),
          l10n_util::GetStringUTF16(IDS_ASH_BIRCH_HIDE_WEATHER_SUGGESTION),
          CreateIconForMenuItem(kForbidIcon));
      break;
    case BirchSuggestionType::kCalendar:
      add_hide_suggestion_item();
      AddItemWithIcon(
          base::to_underlying(CommandId::kHideCalendarSuggestions),
          l10n_util::GetStringUTF16(IDS_ASH_BIRCH_HIDE_CALENDAR_SUGGESTIONS),
          CreateIconForMenuItem(kForbidIcon));
      break;
    case BirchSuggestionType::kDrive:
      add_hide_suggestion_item();
      AddItemWithIcon(
          base::to_underlying(CommandId::kHideDriveSuggestions),
          l10n_util::GetStringUTF16(IDS_ASH_BIRCH_HIDE_DRIVE_SUGGESTIONS),
          CreateIconForMenuItem(kForbidIcon));
      break;
    case BirchSuggestionType::kChromeTab:
      add_hide_suggestion_item();
      AddItemWithIcon(
          base::to_underlying(CommandId::kHideChromeTabSuggestions),
          l10n_util::GetStringUTF16(IDS_ASH_BIRCH_HIDE_CHROME_SUGGESTIONS),
          CreateIconForMenuItem(kForbidIcon));
      break;
    case BirchSuggestionType::kMedia:
      // There's no "Hide this suggestion" for media because the media URLs we
      // have in the BirchItem are very generic (e.g. "youtube.com"), so hiding
      // "this" suggestion would hide all media from a given site.
      AddItemWithIcon(
          base::to_underlying(CommandId::kHideMediaSuggestions),
          l10n_util::GetStringUTF16(IDS_ASH_BIRCH_HIDE_MEDIA_SUGGESTIONS),
          CreateIconForMenuItem(kForbidIcon));
      break;
    case BirchSuggestionType::kCoral:
      // TODO(zxdan): Localize the strings.
      AddItemWithIcon(
          base::to_underlying(CommandId::kCoralNewDesk),
          l10n_util::GetStringUTF16(IDS_ASH_BIRCH_CORAL_CHIP_MENU_OPEN_GROUP),
          CreateIconForMenuItem(kCoralOpenIcon));
      if (features::IsCoralFeatureEnabled() &&
          !display::Screen::Get()->InTabletMode() &&
          !BirchBarController::Get()->is_informed_restore()) {
        AddItemWithIcon(
            base::to_underlying(CommandId::kCoralSaveForLater),
            l10n_util::GetStringUTF16(IDS_ASH_BIRCH_CORAL_CHIP_MENU_SAVE_GROUP),
            CreateIconForMenuItem(kSaveDeskForLaterIcon));
      }
      AddSeparator(ui::NORMAL_SEPARATOR);
      add_hide_suggestion_item();
      AddItemWithIcon(
          base::to_underlying(CommandId::kHideCoralSuggestions),
          l10n_util::GetStringUTF16(IDS_ASH_BIRCH_CORAL_CHIP_MENU_HIDE_GROUP),
          CreateIconForMenuItem(kForbidIcon));
      break;
    default:
      break;
  }

  AddSubMenuWithIcon(
      base::to_underlying(CommandId::kCustomizeSuggestions),
      l10n_util::GetStringUTF16(IDS_ASH_BIRCH_CUSTOMIZE_SUGGESTIONS),
      sub_menu_model_.get(), CreateIconForMenuItem(kPencilIcon));

  if (chip_type == BirchSuggestionType::kWeather) {
    bool is_celsius = GetPrefService()->GetBoolean(prefs::kBirchUseCelsius);
    AddItem(base::to_underlying(CommandId::kToggleTemperatureUnits),
            l10n_util::GetStringUTF16(
                is_celsius ? IDS_ASH_BIRCH_SHOW_TEMPERATURE_IN_FAHRENHEIT
                           : IDS_ASH_BIRCH_SHOW_TEMPERATURE_IN_CELSIUS));
  }

  // Add feedback menu for Coral.
  if (chip_type == BirchSuggestionType::kCoral &&
      coral_util::IsCoralFeedbackAllowedByPolicy(GetPrefService())) {
    AddSeparator(ui::NORMAL_SEPARATOR);
    AddItemWithIcon(base::to_underlying(CommandId::kProvideFeedback),
                    l10n_util::GetStringUTF16(IDS_ASH_BIRCH_SEND_FEEDBACK),
                    CreateIconForMenuItem(kFeedbackIcon));
  }
}

BirchChipContextMenuModel::~BirchChipContextMenuModel() = default;

}  // namespace ash
